import type { NextRequest } from "next/server"
import { gameStore } from "@/lib/game-store"

export async function GET(request: NextRequest, { params }: { params: { gameId: string } }) {
  const gameId = params.gameId

  const stream = new ReadableStream({
    start(controller) {
      // Send initial game state
      const game = gameStore.getGame(gameId)
      if (game) {
        controller.enqueue(`data: ${JSON.stringify(game)}\n\n`)
      }

      // Subscribe to updates
      const unsubscribe = gameStore.subscribe(gameId, (updatedGame) => {
        try {
          controller.enqueue(`data: ${JSON.stringify(updatedGame)}\n\n`)
        } catch (error) {
          // Client disconnected
          unsubscribe()
        }
      })

      // Cleanup on close
      request.signal.addEventListener("abort", () => {
        unsubscribe()
        controller.close()
      })

      // Send heartbeat every 30 seconds
      const heartbeat = setInterval(() => {
        try {
          controller.enqueue(`data: {"type":"heartbeat"}\n\n`)
        } catch (error) {
          clearInterval(heartbeat)
          unsubscribe()
        }
      }, 30000)

      request.signal.addEventListener("abort", () => {
        clearInterval(heartbeat)
      })
    },
  })

  return new Response(stream, {
    headers: {
      "Content-Type": "text/event-stream",
      "Cache-Control": "no-cache",
      Connection: "keep-alive",
    },
  })
}
